package cc.lechun.balance.api;

import cc.lechun.balance.dto.BalanceChangeDTO;
import cc.lechun.balance.dto.SpecialCardDTO;
import cc.lechun.balance.dto.UserBalanceDTO;
import cc.lechun.balance.dto.UserBalanceDetailDTO;
import cc.lechun.framework.common.vo.BaseJsonVo;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;

/**
 * @Description:用户余额接口
 * @Author hans
 * @Date 2019/7/24 10:33
 * @Version 1.0
 */
@RequestMapping(value = "api")
public interface UserBalanceApi {
    /**
     * 用户余额账户信息
     * 注意：需要redis缓存，注意缓存时效，提供单独用户删除缓存的入口/接口
     * @param customerId
     * @return
     */
    @RequestMapping(value = "getUserBalance", method = {RequestMethod.POST, RequestMethod.GET})
    @ResponseBody
    BaseJsonVo<UserBalanceDTO> getUserBalance(String customerId);

    /**
     * 获取用户余额账户信息和所有变更记录
     * 注意：需要redis缓存，注意缓存时效，提供单独用户删除缓存的入口/接口
     * @param customerId
     * @return
     */
    @RequestMapping(value = "getUserBalanceDetail", method = {RequestMethod.POST, RequestMethod.GET})
    @ResponseBody
    BaseJsonVo<UserBalanceDetailDTO> getUserBalanceDetail(String customerId);

    /**
     * 储值
     * 注意：需要分布式锁，需要考虑一致性和幂等性
     * @param customerId
     * @param orderMainNo
     * @param productId
     * @return
     */
    @RequestMapping(value = "charge", method = {RequestMethod.POST, RequestMethod.GET})
    @ResponseBody
    BaseJsonVo<BalanceChangeDTO> charge(String customerId, String orderMainNo, String productId);

    /**
     * 礼品卡兑换
     * 注意：需要分布式锁，需要考虑一致性和幂等性
     * @param customerId
     * @param cardNo
     * @return
     */
    @RequestMapping(value = "cardCharge", method = {RequestMethod.POST, RequestMethod.GET})
    @ResponseBody
    BaseJsonVo<BalanceChangeDTO> cardCharge(String customerId, String cardNo);

    /**
     * 订单支付
     * 注意：需要分布式锁，需要考虑一致性和幂等性
     * @param customerId
     * @param tradeNo
     * @return
     */
    @RequestMapping(value = "payOrder", method = {RequestMethod.POST, RequestMethod.GET})
    @ResponseBody
    BaseJsonVo<BalanceChangeDTO> payOrder(String customerId, String orderMainNo, BigDecimal orderAmount);

    /**
     * 取消订单
     * 注意：需要分布式锁，需要考虑一致性和幂等性
     * @param customerId
     * @param tradeNo
     * @return
     */
    @RequestMapping(value = "cancleOrder", method = {RequestMethod.POST, RequestMethod.GET})
    @ResponseBody
    BaseJsonVo<BalanceChangeDTO> cancleOrder(String customerId, String tradeNo);

    /**
     * 订单退款
     * 注意：需要分布式锁，需要考虑一致性和幂等性
     * @param customerId
     * @param tradeNo
     * @param refundAmount
     * @param operator
     * @return
     */
    @RequestMapping(value = "refundOrder", method = {RequestMethod.POST, RequestMethod.GET})
    @ResponseBody
    BaseJsonVo<BalanceChangeDTO> refundOrder(String customerId, String tradeNo, BigDecimal refundAmount, String operator);

    /**
     * 储值退款
     * 注意：需要分布式锁，需要考虑一致性和幂等性
     * @param customerId
     * @param tradeNo
     * @param operator
     * @return
     */
    @RequestMapping(value = "refundCharge", method = {RequestMethod.POST, RequestMethod.GET})
    @ResponseBody
    BaseJsonVo<BalanceChangeDTO> refundCharge(String customerId, String tradeNo, String operator);

    /**
     * 活动赠送
     * 注意：需要分布式锁，需要考虑一致性和幂等性
     * @param customerId            ：
     * @param increaseAmount        ：
     * @param detailSubTypeDesc     ：活动标识，活动概要，几个字描述一下是啥活动就中
     * @param notice                ：是否发送模板消息，默认不发送
     * @param noticeText            ：模板消息通知文案
     * @return
     */
    @RequestMapping(value = "activeCharge", method = {RequestMethod.POST, RequestMethod.GET})
    @ResponseBody
    BaseJsonVo<BalanceChangeDTO> activeCharge(String customerId, BigDecimal increaseAmount, String detailSubTypeDesc, Boolean notice, String noticeText, String remark);

    /**
     * 手工增加
     * 注意：需要分布式锁，需要考虑一致性和幂等性
     * @param customerId
     * @param increaseAmount
     * @param operator
     * @param detailSubTypeDesc
     * @return
     */
    @RequestMapping(value = "increaseManual", method = {RequestMethod.POST, RequestMethod.GET})
    @ResponseBody
    BaseJsonVo<BalanceChangeDTO> increaseManual(String customerId, BigDecimal increaseAmount, String operator, String detailSubTypeDesc);

    /**
     * 手工减少
     * 注意：需要分布式锁，需要考虑一致性和幂等性
     * @param customerId
     * @param reduceAmount
     * @param operator
     * @param remark
     * @return
     */
    @RequestMapping(value = "reduceManual", method = {RequestMethod.POST, RequestMethod.GET})
    @ResponseBody
    BaseJsonVo<BalanceChangeDTO> reduceManual(String customerId, BigDecimal reduceAmount, String operator, String remark);






    /**
     * 次卡点卡：购买定向专用储值卡
     * 注意：需要分布式锁，需要考虑一致性和幂等性
     * @param customerId
     * @param orderMainNo
     * @return
     */
    @RequestMapping(value = "purchaseSpecialCard", method = {RequestMethod.POST, RequestMethod.GET})
    @ResponseBody
    BaseJsonVo<SpecialCardDTO> purchaseSpecialCard(String customerId, String orderMainNo, BigDecimal chargeAmount, Integer batchType, Integer cardType, BigDecimal saleValue, BigDecimal rate);

    /**
     * 点卡：获取用户定向卡剩余点数
     * 注意：需要分布式锁，需要考虑一致性和幂等性
     * @param customerId
     * @param cardType
     * @return
     */
    @RequestMapping(value = "getSpecialTotal", method = {RequestMethod.POST, RequestMethod.GET})
    @ResponseBody
    BaseJsonVo<BigDecimal> getSpecialTotal(String customerId, Integer cardType);

    /**
     * 次卡点卡：获取用户某个定向卡
     * 注意：需要分布式锁，需要考虑一致性和幂等性
     * @param cardNo
     * @return
     */
    @RequestMapping(value = "getSpecialCardRemain", method = {RequestMethod.POST, RequestMethod.GET})
    @ResponseBody
    Map<String,BigDecimal> getSpecialCardRemain(String cardNo);

    /**
     * 订单支付--使用定向卡
     * 注意：需要分布式锁，需要考虑一致性和幂等性
     * @param customerId
     * @param tradeNo
     * @return
     */
    @RequestMapping(value = "payOrderWithSpecialCard", method = {RequestMethod.POST, RequestMethod.GET})
    @ResponseBody
    BaseJsonVo<BalanceChangeDTO> payOrderWithSpecialCard(String customerId, String orderMainNo, String cardNo, BigDecimal orderAmount, Integer cardType);

    /**
     * 定向卡点卡取消订单返还点数
     * 注意：需要分布式锁，需要考虑一致性和幂等性
     * @param orderMainNo
     * @param cancelAmount
     * @param operator
     * @return
     */
    @RequestMapping(value = "cancelDirectionCardOrder", method = {RequestMethod.POST, RequestMethod.GET})
    @ResponseBody
    BaseJsonVo<BalanceChangeDTO> cancelDirectionCardOrder(String orderMainNo, BigDecimal cancelAmount, String operator);

    /**
     * 定向卡次卡订单退款
     * 注意：需要分布式锁，需要考虑一致性和幂等性
     * @param cardNo
     * @param orderMainNo
     * @param refundAmount
     * @param operator
     * @return
     */
    @RequestMapping(value = "cancelSpecialCardOrder", method = {RequestMethod.POST, RequestMethod.GET})
    @ResponseBody
    BaseJsonVo<BalanceChangeDTO> cancelSpecialCardOrder(String cardNo, String orderMainNo, BigDecimal refundAmount, String operator);

    /**
     * 定向卡订单退款
     * 注意：需要分布式锁，需要考虑一致性和幂等性
     * @param cardNo
     * @param orderMainNo
     * @param refundAmount
     * @param operator
     * @return
     */
    @RequestMapping(value = "refundSpecialCardOrder", method = {RequestMethod.POST, RequestMethod.GET})
    @ResponseBody
    BaseJsonVo<BalanceChangeDTO> refundSpecialCardOrder(String cardNo, String orderMainNo, BigDecimal refundAmount, String operator);

    /**
     * 实物单查询最初金额来源，如果是卡就再追查到余额
     * @param orderMainNo
     * @return
     */
    @RequestMapping(value = "getOrderPayDetail", method = {RequestMethod.POST, RequestMethod.GET})
    @ResponseBody
    BaseJsonVo<Map<String,BigDecimal>> getOrderPayDetail(String orderMainNo);
}
