/*
 * Copyright The OpenTelemetry Authors
 * SPDX-License-Identifier: Apache-2.0
 */

package com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.metrics;

import com.aliyun.openservices.ons.shaded.io.opentelemetry.api.metrics.BoundDoubleValueRecorder;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.api.metrics.DoubleValueRecorder;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.api.metrics.DoubleValueRecorderBuilder;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.api.metrics.common.Labels;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.metrics.aggregator.AggregatorHandle;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.metrics.common.BiFunction;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.metrics.common.InstrumentDescriptor;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.metrics.common.InstrumentType;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.metrics.common.InstrumentValueType;

final class DoubleValueRecorderSdk extends AbstractSynchronousInstrument
    implements DoubleValueRecorder {

  private DoubleValueRecorderSdk(
      InstrumentDescriptor descriptor, SynchronousInstrumentAccumulator<?> accumulator) {
    super(descriptor, accumulator);
  }

  @Override
  public void record(double value, Labels labels) {
    AggregatorHandle<?> aggregatorHandle = acquireHandle(labels);
    try {
      aggregatorHandle.recordDouble(value);
    } finally {
      aggregatorHandle.release();
    }
  }

  @Override
  public void record(double value) {
    record(value, Labels.empty());
  }

  @Override
  public BoundDoubleValueRecorder bind(Labels labels) {
    return new BoundInstrument(acquireHandle(labels));
  }

  static final class BoundInstrument implements BoundDoubleValueRecorder {
    private final AggregatorHandle<?> aggregatorHandle;

    BoundInstrument(AggregatorHandle<?> aggregatorHandle) {
      this.aggregatorHandle = aggregatorHandle;
    }

    @Override
    public void record(double value) {
      aggregatorHandle.recordDouble(value);
    }

    @Override
    public void unbind() {
      aggregatorHandle.release();
    }
  }

  static final class Builder
      extends AbstractSynchronousInstrumentBuilder<DoubleValueRecorderSdk.Builder>
      implements DoubleValueRecorderBuilder {

    Builder(
        String name,
        MeterProviderSharedState meterProviderSharedState,
        MeterSharedState meterSharedState) {
      super(
          name,
          InstrumentType.VALUE_RECORDER,
          InstrumentValueType.DOUBLE,
          meterProviderSharedState,
          meterSharedState);
    }

    @Override
    Builder getThis() {
      return this;
    }

    @Override
    public DoubleValueRecorderSdk build() {
      return buildInstrument(
          new BiFunction<InstrumentDescriptor, SynchronousInstrumentAccumulator<?>, DoubleValueRecorderSdk>() {
            @Override
            public DoubleValueRecorderSdk apply(InstrumentDescriptor instrumentDescriptor,
                SynchronousInstrumentAccumulator<?> synchronousInstrumentAccumulator) {
              return new DoubleValueRecorderSdk(instrumentDescriptor, synchronousInstrumentAccumulator);
            }
          });
    }
  }
}
