/*
 * Copyright The OpenTelemetry Authors
 * SPDX-License-Identifier: Apache-2.0
 */

package com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.trace;

import com.aliyun.openservices.ons.shaded.io.opentelemetry.api.trace.SpanBuilder;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.api.trace.Tracer;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.api.trace.TracerProvider;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.common.InstrumentationLibraryInfo;

/** {@link SdkTracer} is SDK implementation of {@link Tracer}. */
final class SdkTracer implements Tracer {
  static final String FALLBACK_SPAN_NAME = "<unspecified span name>";

  private final TracerSharedState sharedState;
  private final InstrumentationLibraryInfo instrumentationLibraryInfo;

  SdkTracer(TracerSharedState sharedState, InstrumentationLibraryInfo instrumentationLibraryInfo) {
    this.sharedState = sharedState;
    this.instrumentationLibraryInfo = instrumentationLibraryInfo;
  }

  @Override
  public SpanBuilder spanBuilder(String spanName) {
    if (spanName == null || spanName.trim().isEmpty()) {
      spanName = FALLBACK_SPAN_NAME;
    }
    if (sharedState.hasBeenShutdown()) {
      return TracerProvider.noop()
          .get(instrumentationLibraryInfo.getName(), instrumentationLibraryInfo.getVersion())
          .spanBuilder(spanName);
    }
    return new SdkSpanBuilder(
        spanName, instrumentationLibraryInfo, sharedState, sharedState.getSpanLimits());
  }

  /**
   * Returns the instrumentation library specified when creating the tracer.
   *
   * @return an instance of {@link InstrumentationLibraryInfo}
   */
  InstrumentationLibraryInfo getInstrumentationLibraryInfo() {
    return instrumentationLibraryInfo;
  }
}
