/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.impl.consumer;

import com.aliyun.openservices.ons.shaded.com.google.common.base.Stopwatch;
import java.util.List;
import java.util.concurrent.Callable;
import com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.consumer.ConsumeContext;
import com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.consumer.ConsumeStatus;
import com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.consumer.listener.MessageListener;
import com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.message.MessageExt;
import com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.message.MessageHookPoint;
import com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.message.MessageHookPointStatus;
import com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.message.MessageInterceptor;
import com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.message.MessageInterceptorContext;
import com.aliyun.openservices.ons.shaded.org.slf4j.Logger;
import com.aliyun.openservices.ons.shaded.org.slf4j.LoggerFactory;

public class ConsumeTask implements Callable<ConsumeStatus> {
    private static final Logger log = LoggerFactory.getLogger(ConsumeTask.class);

    private final MessageInterceptor interceptor;
    private final MessageListener messageListener;
    private final List<MessageExt> messageExtList;

    public ConsumeTask(MessageInterceptor interceptor, MessageListener messageListener,
                       List<MessageExt> messageExtList) {
        this.interceptor = interceptor;
        this.messageListener = messageListener;
        this.messageExtList = messageExtList;
    }

    @Override
    public ConsumeStatus call() {
        // intercept before message consumption.
        for (MessageExt messageExt : messageExtList) {
            final int attempt = messageExt.getDeliveryAttempt();
            final MessageInterceptorContext context = MessageInterceptorContext.builder()
                                                                               .setTopic(messageExt.getTopic())
                                                                               .setAttempt(attempt).build();
            interceptor.intercept(MessageHookPoint.PRE_MESSAGE_CONSUMPTION, messageExt, context);
        }

        ConsumeStatus status;

        final ConsumeContext consumeContext = new ConsumeContext();
        final Stopwatch stopwatch = Stopwatch.createStarted();
        try {
            status = messageListener.consume(messageExtList, consumeContext);
        } catch (Throwable t) {
            status = ConsumeStatus.ERROR;
            log.error("Message listener raised an exception while consuming messages.", t);
        }
        if (null == status) {
            log.error("Message listener returns NPE for consume status");
            status = ConsumeStatus.ERROR;
        }

        // intercept after message consumption.
        final long duration = stopwatch.elapsed(MessageInterceptor.DEFAULT_TIME_UNIT);
        final MessageHookPointStatus hookPointStatus = ConsumeStatus.OK.equals(status) ? MessageHookPointStatus.OK :
                                                       MessageHookPointStatus.ERROR;
        final int batchSize = messageExtList.size();
        for (MessageExt messageExt : messageExtList) {
            final int attempt = messageExt.getDeliveryAttempt();
            final MessageInterceptorContext context = MessageInterceptorContext.builder()
                                                                               .setTopic(messageExt.getTopic())
                                                                               .setAttempt(attempt)
                                                                               .setDuration(duration)
                                                                               .setBatchSize(batchSize)
                                                                               .setStatus(hookPointStatus).build();
            interceptor.intercept(MessageHookPoint.POST_MESSAGE_CONSUMPTION, messageExt, context);
        }
        return status;
    }
}
