/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.trace;

import com.aliyun.openservices.ons.shaded.grpc.netty.GrpcSslContexts;
import com.aliyun.openservices.ons.shaded.grpc.netty.NettyChannelBuilder;
import com.aliyun.openservices.ons.shaded.io.grpc.netty.shaded.io.netty.handler.ssl.SslContext;
import com.aliyun.openservices.ons.shaded.io.grpc.netty.shaded.io.netty.handler.ssl.util.InsecureTrustManagerFactory;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.api.trace.Span;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.api.trace.Tracer;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.exporter.otlp.trace.OtlpGrpcSpanExporter;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.OpenTelemetrySdk;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.resources.Resource;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.trace.SdkTracerProvider;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.trace.export.BatchSpanProcessor;
import java.net.InetSocketAddress;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;
import com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.impl.ClientImpl;
import com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.remoting.AuthInterceptor;
import com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.remoting.IpNameResolverFactory;
import com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.route.Endpoints;
import com.aliyun.openservices.ons.shaded.org.apache.rocketmq.utility.MetadataUtils;
import com.aliyun.openservices.ons.shaded.org.slf4j.Logger;
import com.aliyun.openservices.ons.shaded.org.slf4j.LoggerFactory;

public class MessageTracer {
    private static final Logger log = LoggerFactory.getLogger(MessageTracer.class);

    /**
     * Name for tracer. See <a href="https://opentelemetry.io">OpenTelemetry</a> for more details.
     */
    private static final String TRACER_INSTRUMENTATION_NAME = "org.apache.rocketmq.message";

    /**
     * Delay interval between two consecutive trace span exports to collector. See
     * <a href="https://opentelemetry.io">OpenTelemetry</a> for more details.
     */
    private static final long TRACE_EXPORTER_SCHEDULE_DELAY_MILLIS = 500L;

    /**
     * Maximum time to wait for the collector to process an exported batch of spans. See
     * <a href="https://opentelemetry.io">OpenTelemetry</a> for more details.
     */
    private static final long TRACE_EXPORTER_RPC_TIMEOUT_MILLIS = 3 * 1000L;

    /**
     * Maximum batch size for every export of span, must be smaller than {@link #TRACE_EXPORTER_MAX_QUEUE_SIZE}.
     * See <a href="https://opentelemetry.io">OpenTelemetry</a> for more details.
     */
    private static final int TRACE_EXPORTER_BATCH_SIZE = 2048;

    /**
     * Maximum number of {@link Span} that are kept in the queue before start dropping. See
     * <a href="https://opentelemetry.io">OpenTelemetry</a> for more details.
     */
    private static final int TRACE_EXPORTER_MAX_QUEUE_SIZE = 16384;

    private final ClientImpl clientImpl;

    private volatile Tracer tracer;
    private volatile Endpoints traceEndpoints;
    private volatile SdkTracerProvider tracerProvider;

    public MessageTracer(ClientImpl clientImpl) {
        this.clientImpl = clientImpl;
    }

    public void init() {
        if (clientImpl.isTracingEnabled()) {
            final TracingMessageInterceptor interceptor = new TracingMessageInterceptor(this);
            clientImpl.registerMessageInterceptor(interceptor);
        }
    }

    @SuppressWarnings("deprecation")
    public synchronized void refresh() {
        if (!clientImpl.isTracingEnabled()) {
            return;
        }
        final List<Endpoints> candidates = clientImpl.getTraceCandidates();
        final String clientId = clientImpl.getId();
        if (candidates.isEmpty()) {
            log.warn("No available message trace endpoints, clientId={}, existed endpoints={}", clientId,
                     traceEndpoints);
            return;
        }
        if (null != traceEndpoints && candidates.contains(traceEndpoints)) {
            log.info("Message trace exporter endpoints remains the same, clientId={}, endpoints={}", clientId,
                     traceEndpoints);
            return;
        }
        try {
            Collections.shuffle(candidates);
            // pick up tracing rpc target randomly.
            final Endpoints newTraceEndpoints = candidates.iterator().next();
            final SslContext sslContext = GrpcSslContexts.forClient().trustManager(InsecureTrustManagerFactory.INSTANCE)
                                                         .build();

            final NettyChannelBuilder channelBuilder =
                    NettyChannelBuilder.forTarget(newTraceEndpoints.getFacade())
                                       .sslContext(sslContext)
                                       .intercept(new AuthInterceptor(clientImpl));

            final List<InetSocketAddress> socketAddresses = newTraceEndpoints.toSocketAddresses();
            // if scheme is not domain.
            if (null != socketAddresses) {
                IpNameResolverFactory tracingResolverFactory = new IpNameResolverFactory(socketAddresses);
                channelBuilder.nameResolverFactory(tracingResolverFactory);
            }

            OtlpGrpcSpanExporter exporter =
                    OtlpGrpcSpanExporter.builder().setChannel(channelBuilder.build())
                                        .setTimeout(TRACE_EXPORTER_RPC_TIMEOUT_MILLIS, TimeUnit.MILLISECONDS).build();

            BatchSpanProcessor spanProcessor =
                    BatchSpanProcessor.builder(exporter)
                                      .setScheduleDelay(TRACE_EXPORTER_SCHEDULE_DELAY_MILLIS, TimeUnit.MILLISECONDS)
                                      .setMaxExportBatchSize(TRACE_EXPORTER_BATCH_SIZE)
                                      .setMaxQueueSize(TRACE_EXPORTER_MAX_QUEUE_SIZE)
                                      .build();
            if (null != tracerProvider) {
                tracerProvider.shutdown();
            }
            final Resource resource = TraceResource.get();
            tracerProvider = SdkTracerProvider.builder().addSpanProcessor(spanProcessor)
                                              .setResource(resource).build();
            OpenTelemetrySdk openTelemetry = OpenTelemetrySdk.builder().setTracerProvider(tracerProvider).build();
            // Fake instrumentation name and version here, tracing related code would be contributed to openTelemetry.
            tracer = openTelemetry.getTracer(TRACER_INSTRUMENTATION_NAME, MetadataUtils.getVersion());
            log.info("Message trace exporter endpoints is updated, clientId={}, {} => {}", clientId,
                     traceEndpoints, newTraceEndpoints);
            traceEndpoints = newTraceEndpoints;
        } catch (Throwable t) {
            log.error("Exception raised while refreshing tracer, clientId={}", clientId, t);
        }
    }

    /**
     * Shutdown the tracer. <strong>Must be called, or the residual spans would not be exported.</strong>
     */
    public void shutdown() {
        log.info("Begin to shutdown the message tracer, clientId={}", clientImpl.getId());
        if (null != tracerProvider) {
            tracerProvider.shutdown();
        }
        log.info("Shutdown the message tracer successfully, clientId={}", clientImpl.getId());
    }

    public Tracer getTracer() {
        return tracer;
    }

    public ClientImpl getClientImpl() {
        return clientImpl;
    }
}
