/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-mqtt-server-js/mqtt_client */
var utils = require('vertx-js/util/utils');
var MqttConnAckMessage = require('vertx-mqtt-server-js/mqtt_conn_ack_message');
var Buffer = require('vertx-js/buffer');
var Vertx = require('vertx-js/vertx');
var MqttPublishMessage = require('vertx-mqtt-server-js/mqtt_publish_message');
var MqttSubAckMessage = require('vertx-mqtt-server-js/mqtt_sub_ack_message');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JMqttClient = Java.type('io.vertx.mqtt.MqttClient');
var MqttClientOptions = Java.type('io.vertx.mqtt.MqttClientOptions');

/**
 An MQTT client

 @class
*/
var MqttClient = function(j_val) {

  var j_mqttClient = j_val;
  var that = this;

  /**
   Connects to an MQTT server calling connectHandler after connection

   @public
   @param connectHandler {function} handler called when the asynchronous connect call ends 
   @return {MqttClient} current MQTT client instance
   */
  this.connect = function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_mqttClient["connect()"]();
      return that;
    }  else if (__args.length === 1 && typeof __args[0] === 'function') {
      j_mqttClient["connect(io.vertx.core.Handler)"](function(ar) {
      if (ar.succeeded()) {
        __args[0](utils.convReturnVertxGen(MqttConnAckMessage, ar.result()), null);
      } else {
        __args[0](null, ar.cause());
      }
    });
      return that;
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Disconnects from the MQTT server calling disconnectHandler after disconnection

   @public
   @param disconnectHandler {function} handler called when asynchronous disconnect call ends 
   @return {MqttClient} current MQTT client instance
   */
  this.disconnect = function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_mqttClient["disconnect()"]();
      return that;
    }  else if (__args.length === 1 && typeof __args[0] === 'function') {
      j_mqttClient["disconnect(io.vertx.core.Handler)"](function(ar) {
      if (ar.succeeded()) {
        __args[0](null, null);
      } else {
        __args[0](null, ar.cause());
      }
    });
      return that;
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Sends the PUBLISH message to the remote MQTT server

   @public
   @param topic {string} topic on which the message is published 
   @param payload {Buffer} message payload 
   @param qosLevel {Object} QoS level 
   @param isDup {boolean} if the message is a duplicate 
   @param isRetain {boolean} if the message needs to be retained 
   @param publishSentHandler {function} handler called after PUBLISH packet sent with packetid (not when QoS 0) 
   @return {MqttClient} current MQTT client instance
   */
  this.publish = function() {
    var __args = arguments;
    if (__args.length === 5 && typeof __args[0] === 'string' && typeof __args[1] === 'object' && __args[1]._jdel && typeof __args[2] === 'string' && typeof __args[3] ==='boolean' && typeof __args[4] ==='boolean') {
      j_mqttClient["publish(java.lang.String,io.vertx.core.buffer.Buffer,io.netty.handler.codec.mqtt.MqttQoS,boolean,boolean)"](__args[0], __args[1]._jdel, io.netty.handler.codec.mqtt.MqttQoS.valueOf(__args[2]), __args[3], __args[4]);
      return that;
    }  else if (__args.length === 6 && typeof __args[0] === 'string' && typeof __args[1] === 'object' && __args[1]._jdel && typeof __args[2] === 'string' && typeof __args[3] ==='boolean' && typeof __args[4] ==='boolean' && typeof __args[5] === 'function') {
      j_mqttClient["publish(java.lang.String,io.vertx.core.buffer.Buffer,io.netty.handler.codec.mqtt.MqttQoS,boolean,boolean,io.vertx.core.Handler)"](__args[0], __args[1]._jdel, io.netty.handler.codec.mqtt.MqttQoS.valueOf(__args[2]), __args[3], __args[4], function(ar) {
      if (ar.succeeded()) {
        __args[5](ar.result(), null);
      } else {
        __args[5](null, ar.cause());
      }
    });
      return that;
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Sets handler which will be called each time publish is completed

   @public
   @param publishCompleteHandler {function} handler called with the packetId 
   @return {MqttClient} current MQTT client instance
   */
  this.publishCompleteHandler = function(publishCompleteHandler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_mqttClient["publishCompleteHandler(io.vertx.core.Handler)"](function(jVal) {
      publishCompleteHandler(jVal);
    });
      return that;
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Sets handler which will be called each time server publish something to client

   @public
   @param publishHandler {function} handler to call 
   @return {MqttClient} current MQTT client instance
   */
  this.publishHandler = function(publishHandler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_mqttClient["publishHandler(io.vertx.core.Handler)"](function(jVal) {
      publishHandler(utils.convReturnVertxGen(MqttPublishMessage, jVal));
    });
      return that;
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Sets handler which will be called after SUBACK packet receiving

   @public
   @param subscribeCompleteHandler {function} handler to call. List inside is a granted QoS array 
   @return {MqttClient} current MQTT client instance
   */
  this.subscribeCompleteHandler = function(subscribeCompleteHandler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_mqttClient["subscribeCompleteHandler(io.vertx.core.Handler)"](function(jVal) {
      subscribeCompleteHandler(utils.convReturnVertxGen(MqttSubAckMessage, jVal));
    });
      return that;
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Subscribes to the topic with a specified QoS level

   @public
   @param topic {string} topic you subscribe on 
   @param qos {number} QoS level 
   @param subscribeSentHandler {function} handler called after SUBSCRIBE packet sent with packetid 
   @return {MqttClient} current MQTT client instance
   */
  this.subscribe = function() {
    var __args = arguments;
    if (__args.length === 1 && (typeof __args[0] === 'object' && __args[0] != null)) {
      j_mqttClient["subscribe(java.util.Map)"](__args[0]);
      return that;
    }  else if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] ==='number') {
      j_mqttClient["subscribe(java.lang.String,int)"](__args[0], __args[1]);
      return that;
    }  else if (__args.length === 2 && (typeof __args[0] === 'object' && __args[0] != null) && typeof __args[1] === 'function') {
      j_mqttClient["subscribe(java.util.Map,io.vertx.core.Handler)"](__args[0], function(ar) {
      if (ar.succeeded()) {
        __args[1](ar.result(), null);
      } else {
        __args[1](null, ar.cause());
      }
    });
      return that;
    }  else if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] ==='number' && typeof __args[2] === 'function') {
      j_mqttClient["subscribe(java.lang.String,int,io.vertx.core.Handler)"](__args[0], __args[1], function(ar) {
      if (ar.succeeded()) {
        __args[2](ar.result(), null);
      } else {
        __args[2](null, ar.cause());
      }
    });
      return that;
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Sets handler which will be called after UNSUBACK packet receiving

   @public
   @param unsubscribeCompleteHandler {function} handler to call with the packetid 
   @return {MqttClient} current MQTT client instance
   */
  this.unsubscribeCompleteHandler = function(unsubscribeCompleteHandler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_mqttClient["unsubscribeCompleteHandler(io.vertx.core.Handler)"](function(jVal) {
      unsubscribeCompleteHandler(jVal);
    });
      return that;
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Unsubscribe from receiving messages on given topic

   @public
   @param topic {string} Topic you want to unsubscribe from 
   @param unsubscribeSentHandler {function} handler called after UNSUBSCRIBE packet sent 
   @return {MqttClient} current MQTT client instance
   */
  this.unsubscribe = function() {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_mqttClient["unsubscribe(java.lang.String)"](__args[0]);
      return that;
    }  else if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_mqttClient["unsubscribe(java.lang.String,io.vertx.core.Handler)"](__args[0], function(ar) {
      if (ar.succeeded()) {
        __args[1](ar.result(), null);
      } else {
        __args[1](null, ar.cause());
      }
    });
      return that;
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Sets handler which will be called after PINGRESP packet receiving

   @public
   @param pingResponseHandler {function} handler to call 
   @return {MqttClient} current MQTT client instance
   */
  this.pingResponseHandler = function(pingResponseHandler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_mqttClient["pingResponseHandler(io.vertx.core.Handler)"](pingResponseHandler);
      return that;
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set an exception handler for the client, that will be called when an error happens
   in internal netty structures.
  
   <code>io.netty.handler.codec.DecoderException</code> can be one of the cause

   @public
   @param handler {function} the exception handler 
   @return {MqttClient} current MQTT client instance
   */
  this.exceptionHandler = function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_mqttClient["exceptionHandler(io.vertx.core.Handler)"](function(jVal) {
      handler(utils.convReturnThrowable(jVal));
    });
      return that;
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set a handler that will be called when the connection with server is closed

   @public
   @param closeHandler {function} handler to call 
   @return {MqttClient} current MQTT client instance
   */
  this.closeHandler = function(closeHandler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_mqttClient["closeHandler(io.vertx.core.Handler)"](closeHandler);
      return that;
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   This method is needed by the client in order to avoid server closes the
   connection due to the keep alive timeout if client has no messages to send

   @public

   @return {MqttClient} current MQTT client instance
   */
  this.ping = function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_mqttClient["ping()"]();
      return that;
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {string} the client identifier
   */
  this.clientId = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_mqttClient["clientId()"]();
    } else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_mqttClient;
};

MqttClient._jclass = utils.getJavaClass("io.vertx.mqtt.MqttClient");
MqttClient._jtype = {
  accept: function(obj) {
    return MqttClient._jclass.isInstance(obj._jdel);
  },
  wrap: function(jdel) {
    var obj = Object.create(MqttClient.prototype, {});
    MqttClient.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
MqttClient._create = function(jdel) {
  var obj = Object.create(MqttClient.prototype, {});
  MqttClient.apply(obj, arguments);
  return obj;
}
/**
 Return an MQTT client instance

 @memberof module:vertx-mqtt-server-js/mqtt_client
 @param vertx {Vertx} Vert.x instance 
 @param options {Object} MQTT client options 
 @return {MqttClient} MQTT client instance
 */
MqttClient.create = function() {
  var __args = arguments;
  if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
    return utils.convReturnVertxGen(MqttClient, JMqttClient["create(io.vertx.core.Vertx)"](__args[0]._jdel));
  }else if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && (typeof __args[1] === 'object' && __args[1] != null)) {
    return utils.convReturnVertxGen(MqttClient, JMqttClient["create(io.vertx.core.Vertx,io.vertx.mqtt.MqttClientOptions)"](__args[0]._jdel, __args[1] != null ? new MqttClientOptions(new JsonObject(Java.asJSONCompatible(__args[1]))) : null));
  } else throw new TypeError('function invoked with invalid arguments');
};

module.exports = MqttClient;