package cc.lechun.framework.common.utils.cache;

import com.google.code.ssm.CacheFactory;
import com.google.code.ssm.api.format.SerializationType;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * Created by wangxs on 2017/12/14
 * Description: 缓存类
 */
@Component
public class MemcachedService {

    private static final Logger log = LoggerFactory.getLogger(MemcachedService.class);

    @Autowired
    private CacheFactory cacheFactory;

    /*
        单位秒
        提供方法上注解的方式进行缓存管理
        1,@ReadThroughSingleCache  读取缓存数据 如果不存在把方法的查询结果放到缓存中
            key生成规则
                通过@ParameterValueKeyProvider 注解  如果注解的对象中存在@CacheKeyMethod 注解的方法 调用这个方法生成key否则toString()
        2,InvalidateSingleCache 失效缓存中数据
            key生成规则
                使用 ParameterValueKeyProvider注解时，与ReadThroughSingleCache一致  注解方法参数
                使用 ReturnValueKeyProvider 注解时，key为返回的对象的CacheKeyMethod或toString方法生成  注解方法返回值
     */

    /**
     * 获取缓存
     */
    public Object get(String key) {
        if (StringUtils.isEmpty(key)) {
            throw new IllegalArgumentException("参数错误！");
        }
        try {
            return cacheFactory.getCache().get(key, SerializationType.JAVA);
        } catch (Exception e) {
            log.error("缓存异常", e);
        }
        return null;
    }

    /**
     * 获取缓存
     */
    public Object get(String nameSpace, String key) {
        if (StringUtils.isEmpty(key)) {
            throw new IllegalArgumentException("参数错误！");
        }
        try {
            return  get(joinNameSpaceKey(nameSpace, key));
        } catch (Exception e) {
            log.error("缓存异常", e);
        }
        return null;
    }
    /**
     * 获取缓存
     */
    public Object get(String nameSpace, String... keys) {
        String key=null;
        if(keys!=null&&keys.length>0){
            key= org.apache.commons.lang3.StringUtils.join(keys,"/");
        }
        return this.get(nameSpace, key);
    }

    /**
     * 添加缓存
     * 最大过期时间
     */
    public boolean set( String key, Object obj) {
        try {
            cacheFactory.getCache().set(key, 60 * 60 * 24 * 30, obj, SerializationType.JAVA);
            return true;
        } catch (Exception e) {
            log.error("缓存异常", e);
        }
        return false;
    }

    /**
     * 添加缓存
     * 最大过期时间
     */
    public boolean set(String nameSpace, String key, Object obj) {
        try {
            cacheFactory.getCache().set(joinNameSpaceKey(nameSpace, key), 60 * 60 * 24 * 30, obj, SerializationType.JAVA);
            return true;
        } catch (Exception e) {
            log.error("缓存异常", e);
        }
        return false;
    }

    /**
     * 添加缓存
     */
    public boolean set(String nameSpace, String key, Object obj, final int exp) {
        try {
            cacheFactory.getCache().set(joinNameSpaceKey(nameSpace, key), exp, obj, SerializationType.JAVA);
            return true;
        } catch (Exception e) {
            log.error("缓存异常", e);
        }
        return false;
    }

    /**
     * 删除缓存
     */
    public boolean delete(String key) {
        try {
            return cacheFactory.getCache().delete(key);
        } catch (Exception e) {
            log.error("缓存异常", e);
        }
        return false;
    }

    /**
     * 删除缓存
     */
    public boolean delete(String nameSpace, String key) {
        try {
            return delete(joinNameSpaceKey(nameSpace, key));
        } catch (Exception e) {
            log.error("缓存异常", e);
        }
        return false;
    }

    /**
     * 删除缓存
     */
    public boolean delete(String nameSpace, String... keys) {
        String key=null;
        if(keys!=null&&keys.length>0){
            key= org.apache.commons.lang3.StringUtils.join(keys,"/");
        }
        return this.delete(nameSpace,key);
    }

    /**
     * 原子性计数  此方法加减的值 用这种方法获取  不用get方法获取
     *
     * @param nameSpace
     * @param key
     * @param count     加的数
     * @param def       不存在默认的数
     * @return 加完的数
     */
    public Long incr(String nameSpace, String key, int count, long def) {
        try {
            return cacheFactory.getCache().incr(joinNameSpaceKey(nameSpace, key), count, def);
        } catch (Exception e) {
            log.error("缓存异常", e);
        }
        return null;
    }

    public Long incr(String nameSpace, String key, int count, long def, int exp) {
        try {
            return cacheFactory.getCache().incr(joinNameSpaceKey(nameSpace, key), count, def, exp);
        } catch (Exception e) {
            log.error("缓存异常", e);
        }
        return null;
    }

    /**
     * 减法 不存在为0
     */
    public Long decr(String nameSpace, String key, int count) {
        try {
            return cacheFactory.getCache().decr(joinNameSpaceKey(nameSpace, key), count);
        } catch (Exception e) {
            log.error("缓存异常", e);
        }
        return null;
    }

    /**
     * 指定缓存是否 在指定时间内 超过次数
     * @param nameSpace  缓存命名空间
     * @param key 缓存key
     * @param count 总共次数
     * @param times 指定时间
     * @return
     */
    public boolean isExceedCount(String nameSpace,String key,int count,int times){
        long old_count=incr(nameSpace,key,0,0,times);
        if(old_count>=count){
            //TODO 最好短信或者邮件通知
            return true;
        }else{
            incr(nameSpace,key,1,1);
        }
        return false;
    }

    private String joinNameSpaceKey(String nameSpace, String key) {
        return nameSpace + ":" + key;
    }
}
