/*
 * Decompiled with CFR 0.152.
 */
package org.apache.mahout.math.stats;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Random;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.mahout.common.RandomUtils;
import org.apache.mahout.math.stats.GroupTree;

public class TDigest {
    private Random gen = RandomUtils.getRandom();
    private double compression = 100.0;
    private GroupTree summary = new GroupTree();
    private int count = 0;
    private boolean recordAllData = false;
    public static final int VERBOSE_ENCODING = 1;
    public static final int SMALL_ENCODING = 2;

    public TDigest(double compression) {
        this.compression = compression;
    }

    public void add(double x) {
        this.add(x, 1);
    }

    public void add(double x, int w) {
        Group base = this.createGroup(x, 0);
        this.add(x, w, base);
    }

    private void add(double x, int w, Group base) {
        Group start = this.summary.floor(base);
        if (start == null) {
            start = this.summary.ceiling(base);
        }
        if (start == null) {
            this.summary.add(Group.createWeighted(x, w, base.data()));
            this.count = w;
        } else {
            Group neighbor;
            double z;
            Iterable<Group> neighbors = this.summary.tailSet(start);
            double minDistance = Double.MAX_VALUE;
            int lastNeighbor = 0;
            int i = this.summary.headCount(start);
            Iterator<Group> i$ = neighbors.iterator();
            while (i$.hasNext() && (z = Math.abs((neighbor = i$.next()).mean() - x)) <= minDistance) {
                minDistance = z;
                lastNeighbor = i++;
            }
            Group closest = null;
            int sum = this.summary.headSum(start);
            i = this.summary.headCount(start);
            double n = 1.0;
            for (Group neighbor2 : neighbors) {
                if (i > lastNeighbor) break;
                double z2 = Math.abs(neighbor2.mean() - x);
                double q = ((double)sum + (double)neighbor2.count() / 2.0) / (double)this.count;
                double k = (double)(4 * this.count) * q * (1.0 - q) / this.compression;
                if (z2 == minDistance && (double)(neighbor2.count() + w) <= k) {
                    if (this.gen.nextDouble() < 1.0 / n) {
                        closest = neighbor2;
                    }
                    n += 1.0;
                }
                sum += neighbor2.count();
                ++i;
            }
            if (closest == null) {
                this.summary.add(Group.createWeighted(x, w, base.data()));
            } else {
                this.summary.remove(closest);
                closest.add(x, w, base.data());
                this.summary.add(closest);
            }
            this.count += w;
            if ((double)this.summary.size() > 100.0 * this.compression) {
                this.compress();
            }
        }
    }

    public void add(TDigest other) {
        ArrayList tmp = Lists.newArrayList((Iterable)other.summary);
        Collections.shuffle(tmp);
        for (Group group : tmp) {
            this.add(group.mean(), group.count(), group);
        }
    }

    public static TDigest merge(double compression, Iterable<TDigest> subData) {
        ArrayList elements = Lists.newArrayList(subData);
        int n = Math.max(1, elements.size() / 4);
        TDigest r = new TDigest(compression);
        if (elements.size() > 0 && ((TDigest)elements.get((int)0)).recordAllData) {
            r.recordAllData();
        }
        for (int i = 0; i < elements.size(); i += n) {
            if (n > 1) {
                r.add(TDigest.merge(compression, elements.subList(i, Math.min(i + n, elements.size()))));
                continue;
            }
            r.add((TDigest)elements.get(i));
        }
        return r;
    }

    public void compress() {
        this.compress(this.summary);
    }

    private void compress(GroupTree other) {
        TDigest reduced = new TDigest(this.compression);
        if (this.recordAllData) {
            reduced.recordAllData();
        }
        ArrayList tmp = Lists.newArrayList((Iterable)other);
        Collections.shuffle(tmp);
        for (Group group : tmp) {
            reduced.add(group.mean(), group.count(), group);
        }
        this.summary = reduced.summary;
    }

    public int size() {
        return this.count;
    }

    public double cdf(double x) {
        double left;
        GroupTree values = this.summary;
        if (values.size() == 0) {
            return Double.NaN;
        }
        if (values.size() == 1) {
            return x < values.first().mean() ? 0.0 : 1.0;
        }
        double r = 0.0;
        Iterator<Group> it = values.iterator();
        Group a = it.next();
        Group b = it.next();
        double right = left = (b.mean() - a.mean()) / 2.0;
        while (it.hasNext()) {
            if (x < a.mean() + right) {
                return (r + (double)a.count() * this.interpolate(x, a.mean() - left, a.mean() + right)) / (double)this.count;
            }
            r += (double)a.count();
            a = b;
            b = it.next();
            left = right;
            right = (b.mean() - a.mean()) / 2.0;
        }
        left = right;
        a = b;
        if (x < a.mean() + right) {
            return (r + (double)a.count() * this.interpolate(x, a.mean() - left, a.mean() + right)) / (double)this.count;
        }
        return 1.0;
    }

    public double quantile(double q) {
        double right;
        GroupTree values = this.summary;
        Preconditions.checkArgument((values.size() > 1 ? 1 : 0) != 0);
        Iterator<Group> it = values.iterator();
        Group a = it.next();
        Group b = it.next();
        if (!it.hasNext()) {
            double diff = (b.mean() - a.mean()) / 2.0;
            if (q > 0.75) {
                return b.mean() + diff * (4.0 * q - 3.0);
            }
            return a.mean() + diff * (4.0 * q - 1.0);
        }
        q *= (double)this.count;
        double left = right = (b.mean() - a.mean()) / 2.0;
        if (q <= (double)a.count()) {
            return a.mean() + left * (2.0 * q - (double)a.count()) / (double)a.count();
        }
        double t = a.count();
        while (it.hasNext()) {
            if (t + (double)(b.count() / 2) >= q) {
                return b.mean() - left * 2.0 * (q - t) / (double)b.count();
            }
            if (t + (double)b.count() >= q) {
                return b.mean() + right * 2.0 * (q - t - (double)b.count() / 2.0) / (double)b.count();
            }
            t += (double)b.count();
            a = b;
            b = it.next();
            left = right;
            right = (b.mean() - a.mean()) / 2.0;
        }
        return b.mean() + right;
    }

    public int centroidCount() {
        return this.summary.size();
    }

    public Iterable<? extends Group> centroids() {
        return this.summary;
    }

    public double compression() {
        return this.compression;
    }

    public TDigest recordAllData() {
        this.recordAllData = true;
        return this;
    }

    public int byteSize() {
        return 16 + this.summary.size() * 12;
    }

    public int smallByteSize() {
        int bound = this.byteSize();
        ByteBuffer buf = ByteBuffer.allocate(bound);
        this.asSmallBytes(buf);
        return buf.position();
    }

    public void asBytes(ByteBuffer buf) {
        buf.putInt(1);
        buf.putDouble(this.compression());
        buf.putInt(this.summary.size());
        for (Group group : this.summary) {
            buf.putDouble(group.mean());
        }
        for (Group group : this.summary) {
            buf.putInt(group.count());
        }
    }

    public void asSmallBytes(ByteBuffer buf) {
        buf.putInt(2);
        buf.putDouble(this.compression());
        buf.putInt(this.summary.size());
        double x = 0.0;
        for (Group group : this.summary) {
            double delta = group.mean() - x;
            x = group.mean();
            buf.putFloat((float)delta);
        }
        for (Group group : this.summary) {
            int n = group.count();
            TDigest.encode(buf, n);
        }
    }

    public static void encode(ByteBuffer buf, int n) {
        int k = 0;
        while (n < 0 || n > 127) {
            byte b = (byte)(0x80 | 0x7F & n);
            buf.put(b);
            n >>>= 7;
            Preconditions.checkState((++k < 6 ? 1 : 0) != 0);
        }
        buf.put((byte)n);
    }

    public static int decode(ByteBuffer buf) {
        byte v = buf.get();
        int z = 0x7F & v;
        int shift = 7;
        while ((v & 0x80) != 0) {
            Preconditions.checkState((shift <= 28 ? 1 : 0) != 0);
            v = buf.get();
            z += (v & 0x7F) << shift;
            shift += 7;
        }
        return z;
    }

    public static TDigest fromBytes(ByteBuffer buf) {
        int encoding = buf.getInt();
        if (encoding == 1) {
            int i;
            double compression = buf.getDouble();
            TDigest r = new TDigest(compression);
            int n = buf.getInt();
            double[] means = new double[n];
            for (i = 0; i < n; ++i) {
                means[i] = buf.getDouble();
            }
            for (i = 0; i < n; ++i) {
                r.add(means[i], buf.getInt());
            }
            return r;
        }
        if (encoding == 2) {
            int i;
            double compression = buf.getDouble();
            TDigest r = new TDigest(compression);
            int n = buf.getInt();
            double[] means = new double[n];
            double x = 0.0;
            for (i = 0; i < n; ++i) {
                double delta = buf.getFloat();
                means[i] = x += delta;
            }
            for (i = 0; i < n; ++i) {
                int z = TDigest.decode(buf);
                r.add(means[i], z);
            }
            return r;
        }
        throw new IllegalStateException("Invalid format for serialized histogram");
    }

    private Group createGroup(double mean, int id) {
        return new Group(mean, id, this.recordAllData);
    }

    private double interpolate(double x, double x0, double x1) {
        return (x - x0) / (x1 - x0);
    }

    public static class Group
    implements Comparable<Group> {
        private static final AtomicInteger uniqueCount = new AtomicInteger(1);
        private double centroid = 0.0;
        private int count = 0;
        private int id = uniqueCount.incrementAndGet();
        private List<Double> actualData = null;

        private Group(boolean record) {
            if (record) {
                this.actualData = Lists.newArrayList();
            }
        }

        public Group(double x) {
            this(false);
            this.start(x, uniqueCount.getAndIncrement());
        }

        public Group(double x, int id) {
            this(false);
            this.start(x, id);
        }

        public Group(double x, int id, boolean record) {
            this(record);
            this.start(x, id);
        }

        private void start(double x, int id) {
            this.id = id;
            this.add(x, 1);
        }

        public void add(double x, int w) {
            if (this.actualData != null) {
                this.actualData.add(x);
            }
            this.count += w;
            this.centroid += (double)w * (x - this.centroid) / (double)this.count;
        }

        public double mean() {
            return this.centroid;
        }

        public int count() {
            return this.count;
        }

        public int id() {
            return this.id;
        }

        public String toString() {
            return "Group{centroid=" + this.centroid + ", count=" + this.count + '}';
        }

        public int hashCode() {
            return this.id;
        }

        @Override
        public int compareTo(Group o) {
            int r = Double.compare(this.centroid, o.centroid);
            if (r == 0) {
                r = this.id - o.id;
            }
            return r;
        }

        public Iterable<? extends Double> data() {
            return this.actualData;
        }

        public static Group createWeighted(double x, int w, Iterable<? extends Double> data) {
            Group r = new Group(data != null);
            r.add(x, w, data);
            return r;
        }

        private void add(double x, int w, Iterable<? extends Double> data) {
            if (this.actualData != null) {
                if (data != null) {
                    for (Double d : data) {
                        this.actualData.add(d);
                    }
                } else {
                    this.actualData.add(x);
                }
            }
            this.count += w;
            this.centroid += (double)w * (x - this.centroid) / (double)this.count;
        }
    }
}

