package cc.lechun.framework.common.utils.cache;

import java.io.Serializable;
import java.util.*;
import java.util.concurrent.TimeUnit;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.support.ClassPathXmlApplicationContext;
import org.springframework.data.redis.core.*;
import org.springframework.stereotype.Service;
import redis.clients.jedis.Jedis;

@Service
public class RedisCacheUtil<T>
{
    private static final Long RELEASE_SUCCESS = 1L;
    private static final String LOCK_SUCCESS = "OK";
    private static final String SET_IF_NOT_EXIST = "NX";
    private static final String SET_WITH_EXPIRE_TIME = "EX";//单位“秒”
    private static final String RELEASE_LOCK_SCRIPT = "if redis.call('get', KEYS[1]) == ARGV[1] then return redis.call('del', KEYS[1]) else return 0 end";
    private static final Long DEFAULT_EXPIRE_TIME = 60L;

    @SuppressWarnings("rawtypes")
    @Autowired
    private RedisTemplate redisTemplate;

    /**
     * 批量删除对应的value
     *
     * @param keys
     */
    public void remove(final String... keys) {
        for (String key : keys) {
            remove(key);
        }
    }

    /**
     * 批量删除key
     *
     * @param pattern
     */
    @SuppressWarnings("unchecked")
    public void removePattern(final String pattern) {
        Set<Serializable> keys = redisTemplate.keys(pattern);
        if (keys.size() > 0)
            redisTemplate.delete(keys);
    }

    /**
     * 删除对应的value
     *
     * @param key
     */
    @SuppressWarnings("unchecked")
    public void remove(final String key) {
        if (exists(key)) {
            redisTemplate.delete(key);
        }
    }
    /**
     * 删除对应的value
     *
     * @param keys
     */
    @SuppressWarnings("unchecked")
    public void remove(Set<Serializable> keys ) {
        if (keys.size() > 0)
            redisTemplate.delete(keys);
    }
    /**
     * 判断缓存中是否有对应的value
     *
     * @param key
     * @return
     */
    @SuppressWarnings("unchecked")
    public boolean exists(final String key) {
        return redisTemplate.hasKey(key);
    }

    /**
     * 读取缓存
     *
     * @param key
     * @return
     */
    @SuppressWarnings("unchecked")
    public Object get(final String key) {
        Object result = null;
        ValueOperations<Serializable, Object> operations = redisTemplate.opsForValue();
        result = operations.get(key);
        return result;
    }

    /**
     * 写入缓存
     *
     * @param key
     * @param value
     * @return
     */
    @SuppressWarnings("unchecked")
    public boolean set(final String key, Object value) {
        return set(key,value,0L);
    }

    /**
     * 写入缓存
     *
     * @param key
     * @param value
     * @return
     */
    @SuppressWarnings("unchecked")
    public boolean set(final String key, Object value, Long expireTime) {
        boolean result = false;
        try {
            ValueOperations<Serializable, Object> operations = redisTemplate.opsForValue();
            operations.set(key, value);
            if(expireTime!=0) {
                redisTemplate.expire(key, expireTime, TimeUnit.SECONDS);
            }
            result = true;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return result;
    }

    /**
     * 递减
     * @param key
     * @param delta
     * @return
     */
    public int decrement(String key, int delta) {
        Long value = redisTemplate.opsForValue().increment(key, -delta);
        return value.intValue();
    }

    /**
     * 自增
     * @param key
     * @param delta
     * @return
     */
    public int increment(String key, int delta) {
        Long value = redisTemplate.opsForValue().increment(key, delta);
        return value.intValue();
    }

    /**
     * 缓存基本的对象，Integer、String、实体类等
     * @param key    缓存的键值
     * @param value    缓存的值
     * @return        缓存的对象
     */
    public <T> ValueOperations<String,T> setCacheObject(String key, T value){
        return setCacheObject(key,value,0L);
    }
    /**
     * 缓存基本的对象，Integer、String、实体类等
     * @param key    缓存的键值
     * @param value    缓存的值
     * @return        缓存的对象
     */
    public <T> ValueOperations<String,T> setCacheObject(String key, T value,Long expireTime){
        ValueOperations<String,T> operation = redisTemplate.opsForValue();
        operation.set(key,value);
        if(expireTime!=0) {
            redisTemplate.expire(key, expireTime, TimeUnit.SECONDS);
        }
        return operation;
    }
    /**
     * 获得缓存的基本对象。
     * @param key        缓存键值
     * @return            缓存键值对应的数据
     */
    public <T> T getCacheObject(String key)
    {
        ValueOperations<String,T> operation = redisTemplate.opsForValue();
        return operation.get(key);
    }

    /**
     * 缓存List数据
     * @param key        缓存的键值
     * @param dataList    待缓存的List数据
     * @return            缓存的对象
     */
    public <T> ListOperations<String, T> setCacheList(String key, List<T> dataList){
        return setCacheList(key,dataList,0);
    }
    /**
     * 缓存List数据
     * @param key        缓存的键值
     * @param dataList    待缓存的List数据
     * @return            缓存的对象
     */
    public <T> ListOperations<String, T> setCacheList(String key, List<T> dataList,long expireTime){
        ListOperations listOperation = redisTemplate.opsForList();
        if(null != dataList)
        {
            int size = dataList.size();
            for(int i = 0; i < size ; i ++)
            {
                listOperation.rightPush(key,dataList.get(i));
            }
            if(expireTime!=0){
                redisTemplate.expire(key, expireTime, TimeUnit.SECONDS);
            }
        }
        return listOperation;
    }
    /**
     * 获得缓存的list对象
     * @param key    缓存的键值
     * @return        缓存键值对应的数据
     */
    public <T> List<T> getCacheList(String key){
        List<T> dataList = new ArrayList<T>();
        ListOperations<String,T> listOperation = redisTemplate.opsForList();
        Long size = listOperation.size(key);
        for(int i = 0 ; i < size ; i ++)
        {
            dataList.add((T) listOperation.leftPop(key));
        }

        return dataList;
    }

    /**
     * 缓存Set
     * @param key        缓存键值
     * @param dataSet    缓存的数据
     * @return            缓存数据的对象
     */
    public <T> BoundSetOperations<String,T> setCacheSet(String key, Set<T> dataSet){
        return setCacheSet(key,dataSet,0L);
    }
    /**
     * 缓存Set
     * @param key        缓存键值
     * @param dataSet    缓存的数据
     * @return            缓存数据的对象
     */
    public <T> BoundSetOperations<String,T> setCacheSet(String key, Set<T> dataSet,Long expireTime){
        BoundSetOperations<String,T> setOperation = redisTemplate.boundSetOps(key);
        Iterator<T> it = dataSet.iterator();
        while(it.hasNext())
        {
            setOperation.add(it.next());
        }
        if(expireTime!=0){
            redisTemplate.expire(key, expireTime, TimeUnit.SECONDS);
        }
        return setOperation;
    }

    /**
     * 缓存Set
     * @param key        缓存键值
     * @param ls    缓存的数据
     * @return            缓存数据的对象
     */
    public boolean setCacheSet(String key,T... ls){
        try {
            BoundSetOperations<String, T> setOperation = redisTemplate.boundSetOps(key);
            Long i = setOperation.add(ls);
            if (i == 0) {
                return false;
            }
        }catch (Exception e){
            return false;
        }
        return true;
    }
    /**
     * 缓存Set
     * @param key        缓存键值
     * @param ls    缓存的数据
     * @return            缓存数据的对象
     */
    public boolean setCacheSet(String key,Long expireTime,T... ls){
        try {
            BoundSetOperations<String, T> setOperation = redisTemplate.boundSetOps(key);
            Long i = setOperation.add(ls);
            if(expireTime!=0){
                redisTemplate.expire(key, expireTime, TimeUnit.SECONDS);
            }
            if (i == 0) {
                return false;
            }
        }catch (Exception e){
            return false;
        }
        return true;
    }

    /**
     * 移除缓存Set中的元素
     * @param key        缓存键值
     * @param ls    缓存的数据
     * @return            缓存数据的对象
     */
    public boolean removeCacheSet(String key,T... ls){
        try {
            BoundSetOperations<String, T> setOperation = redisTemplate.boundSetOps(key);
            Long i = setOperation.remove(ls);
            if (i == 0) {
                return false;
            }
        }catch (Exception e){
            return false;
        }
        return true;
    }
    /**
     * 获得缓存的set
     * @param key
     * @return
     */
    public Set<T> getCacheSet(String key){
        Set<T> dataSet = new HashSet<T>();
        BoundSetOperations<String,T> operation = redisTemplate.boundSetOps(key);
        Long size = operation.size();
        for(int i = 0 ; i < size ; i++)
        {
            dataSet.add(operation.pop());
        }
        return dataSet;
    }

    /**
     * 缓存Map
     * @param key
     * @param dataMap
     * @return
     */
    public <T> HashOperations<String,String,T> setCacheMap(String key, Map<String,T> dataMap) {
        return setCacheMap(key,dataMap,0L);
    }
    /**
     * 缓存Map
     * @param key
     * @param dataMap
     * @return
     */
    public <T> HashOperations<String,String,T> setCacheMap(String key, Map<String,T> dataMap,Long expireTime) {
        HashOperations hashOperations = redisTemplate.opsForHash();
        if(null != dataMap){
            for (Map.Entry<String, T> entry : dataMap.entrySet()) {
                hashOperations.put(key,entry.getKey(),entry.getValue());
            }
            if(expireTime!=0){
                redisTemplate.expire(key, expireTime, TimeUnit.SECONDS);
            }
        }
        return hashOperations;
    }

    /**
     * 获得缓存的Map
     * @param key
     * @return
     */
    public <T> Map<String,T> getCacheMap(String key){
        Map<String, T> map = redisTemplate.opsForHash().entries(key);
        return map;
    }

    /**
     * 缓存Map
     * @param key
     * @param dataMap
     * @return
     */
    public <T> HashOperations<String,Integer,T> setCacheIntegerMap(String key, Map<Integer,T> dataMap){
        return setCacheIntegerMap(key,dataMap,0L);
    }
    /**
     * 缓存Map
     * @param key
     * @param dataMap
     * @return
     */
    public <T> HashOperations<String,Integer,T> setCacheIntegerMap(String key, Map<Integer,T> dataMap,Long expireTime){
        HashOperations hashOperations = redisTemplate.opsForHash();
        if(null != dataMap)
        {
            for (Map.Entry<Integer, T> entry : dataMap.entrySet()) {
                hashOperations.put(key,entry.getKey(),entry.getValue());
            }
            if(expireTime!=0){
                redisTemplate.expire(key, expireTime, TimeUnit.SECONDS);
            }
        }
        return hashOperations;
    }

    /**
     * 获得缓存的Map
     * @param key
     * @return
     */
    public <T> Map<Integer,T> getCacheIntegerMap(String key){
        Map<Integer, T> map = redisTemplate.opsForHash().entries(key);
        return map;
    }

    /**
     * 循环等待获取锁
     * @param lockKey
     * @param clientId
     * @param expireTime
     * @param waitTime      :   等待时间，时间秒
     * @return
     */
    public Boolean waitLock(String lockKey, String clientId, long expireTime, long waitTime) {
        long nanoWaitTime = TimeUnit.SECONDS.toNanos(waitTime);
        long now = System.nanoTime();
        while (System.nanoTime() - now < nanoWaitTime){
            if(lock(lockKey, clientId, expireTime))
                return Boolean.TRUE;
        }
        return Boolean.FALSE;
    }

    /**
     * 默认加锁60秒，即当前任务完成后才能
     * @param lockKey
     * @param clientId
     * @return
     */
    public Boolean lock(String lockKey, String clientId) {
        return lock(lockKey, clientId, DEFAULT_EXPIRE_TIME);
    }

    /**
     * 该加锁方法仅针对单实例 Redis 可实现分布式加锁
     * 对于 Redis 集群则无法使用
     * 支持重复，线程安全
     *
     * @param lockKey       :   加锁键
     * @param clientId      :   加锁客户端唯一标识
     * @param expireTime    :   锁过期时间(秒)，最长不超过300秒
     * @return
     */
    public Boolean lock(String lockKey, String clientId, long expireTime) {
        return (Boolean) redisTemplate.execute((RedisCallback<Boolean>) redisConnection -> {
            Jedis jedis = (Jedis) redisConnection.getNativeConnection();
            String result = jedis.set(lockKey, clientId, SET_IF_NOT_EXIST, SET_WITH_EXPIRE_TIME,
                    expireTime <= 0 ? 1 : expireTime > 300L ? 300L : expireTime);
            if (LOCK_SUCCESS.equals(result)) {
                return Boolean.TRUE;
            }
            return Boolean.FALSE;
        });
    }

    /**
     * 与 lock 相对应，用作释放锁
     * @param lockKey   :   加锁键
     * @param clientId  :   加锁客户端唯一标识
     * @return
     */
    public Boolean releaseLock(String lockKey, String clientId) {
        return (Boolean) redisTemplate.execute((RedisCallback<Boolean>) redisConnection -> {
            Jedis jedis = (Jedis) redisConnection.getNativeConnection();
            Object result = jedis.eval(RELEASE_LOCK_SCRIPT, Collections.singletonList(lockKey),
                    Collections.singletonList(clientId));
            if (RELEASE_SUCCESS.equals(result)) {
                return Boolean.TRUE;
            }
            return Boolean.FALSE;
        });
    }
    /**
     * 得到批量key
     *
     * @param pattern
     */
    @SuppressWarnings("unchecked")
    public Set<Serializable> getKeysPattern(final String pattern) {
        return redisTemplate.keys(pattern);
    }

}
