package com.aliyun.manager;

import com.aliyun.Context;
import com.aliyun.bean.config.ToolkitDeployConfig;
import com.aliyun.bean.config.ToolkitPackageConfig;
import com.aliyun.utils.CommonUtils;
import com.aliyun.utils.HttpUtils;
import com.aliyuncs.DefaultAcsClient;
import com.aliyuncs.edas.model.v20170801.ListApplicationRequest;
import com.aliyuncs.edas.model.v20170801.ListApplicationResponse;
import com.aliyuncs.sae.model.v20190506.ListApplicationsRequest;
import com.aliyuncs.sae.model.v20190506.ListApplicationsResponse;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;

import java.text.SimpleDateFormat;
import java.util.Date;

import static com.aliyun.enums.Constants.*;

public class AppConfigManager {

    private Log logger;

    public AppConfigManager() {
        this.logger = Context.getLogger();
    }

    public void complete(
            ToolkitDeployConfig deployConfig,
            ToolkitPackageConfig packageConfig,
            MavenProject project, DefaultAcsClient acsClient) throws Exception {
        fixAppId(deployConfig, acsClient);

        //detectClusterType(config, acsClient);
        String type = deployConfig.getSpec().getType();
        if (SERVERLESS.equals(type)) {
//            detectServerlessPackageType(deployConfig, packageConfig, acsClient);
            //validateServerlessConfigParam(deployConfig);
            setDefaultConfigForServerless(deployConfig, project);
        } else {
            //validateNonServerlessConfigParam(deployConfig);
            setDefaultConfigForNonServerless(deployConfig, packageConfig, project);
        }
    }

    private void fixAppId(ToolkitDeployConfig deployConfig, DefaultAcsClient acsClient) throws Exception {
        if (deployConfig.getSpec().getTarget() == null) {
            deployConfig.getSpec().setTarget(new ToolkitDeployConfig.Target());
        }
        // 如果是sae应用，只有appId不存在才去fix
        if (!SERVERLESS.equals(deployConfig.getSpec().getType())) {
            tryToFixAppId(deployConfig.getSpec(), acsClient);
        } else if (deployConfig.getSpec().getTarget().getAppId() == null) {
            tryToFixAppIdServerless(deployConfig.getSpec(), acsClient);
        }

        if (deployConfig.getSpec().getTarget().getAppId() == null) {
            throw new Exception("'appId' is not set");
        }
    }


    private void tryToFixAppId(ToolkitDeployConfig.Spec spec, DefaultAcsClient client) throws Exception {
        if ((spec.getTarget().getNamespaceId() == null || spec.getTarget().getAppName() == null)
            && spec.getTarget().getAppId() == null) {
            return;
        }

        logger.info(String.format("fix appId(%s)", spec.getTarget().getAppId()));
        // 新增根据appId判断是ecs还是k8s集群功能
        ListApplicationRequest request = new ListApplicationRequest();
        ListApplicationResponse response = HttpUtils.getAcsResponseIfRetry(client, request);
        if (response.getCode() != 200) {
            return;
        }

        for (ListApplicationResponse.Application application: response.getApplicationList()) {
            if ((application.getRegionId().equals(spec.getTarget().getNamespaceId())
                    && application.getName().equals(spec.getTarget().getAppName()))
                || application.getAppId().equals(spec.getTarget().getAppId())) {
                logger.info(String.format("Use app id for app(%s) in namespace(%s)",
                    application.getName(), application.getRegionId()));
                spec.getTarget().setAppId(application.getAppId());
                ObjectMapper jsonMapper = new ObjectMapper();
                jsonMapper.configure(SerializationFeature.INDENT_OUTPUT,true);
                logger.info(String.format("fix appId application = (%s) \n",jsonMapper.writeValueAsString(application)));
                // 识别k8s集群
                if (application.getClusterType() == KUBERNETES_CLUSTER) {
                    spec.setType(KUBERNETES);
                }
                return;
            }
        }
    }

    private void tryToFixAppIdServerless(ToolkitDeployConfig.Spec spec, DefaultAcsClient client) throws Exception {
        if (spec.getTarget().getNamespaceId() == null || spec.getTarget().getAppName() == null) {
            return;
        }

        ListApplicationsRequest request = new ListApplicationsRequest();
        request.setAppName(spec.getTarget().getAppName());
        ListApplicationsResponse response = com.aliyun.utils.HttpUtils.getAcsResponseIfRetry(client, request);
        if (!"200".equals(response.getCode())) {
            return;
        }
        for (ListApplicationsResponse.Data.Application application: response.getData().getApplications()) {
            String nameSpaceId;
            if (application.getNamespaceId() != null) {
                nameSpaceId = application.getNamespaceId();
            } else {
                nameSpaceId = application.getRegionId();
            }
            if (nameSpaceId.equals(spec.getTarget().getNamespaceId())
                && application.getAppName().equals(spec.getTarget().getAppName())) {
                logger.info(String.format("Use app id for app(%s) in namespace(%s)",
                    spec.getTarget().getAppName(), spec.getTarget().getNamespaceId()));
                spec.getTarget().setAppId(application.getAppId());
                return;
            }
        }
    }


//    private void validateServerlessConfigParam(ToolkitDeployConfig config) throws Exception {
//
//    }
//
//    private void validateNonServerlessConfigParam(ToolkitDeployConfig deployConfig) throws Exception {
//        //DefaultConfigBean.App app = config.getApp();
//        if (deployConfig.getSpec().getDeployType() != null
//                && !"url".equals(deployConfig.getSpec().getDeployType())) {
//            throw new Exception("Currently this plugin only support 'url' deploy type");
//        }
//    }

    private void setDefaultConfigForServerless(
            ToolkitDeployConfig deployConfig,
            MavenProject project) {
        //DefaultConfigBean.App app = config.getApp();
        deployConfig.getSpec().setStageTimeout(300);
        deployConfig.getSpec().setServiceStageTimeout(300);
        deployConfig.getSpec().setInstanceStageTimeout(300);

        if (deployConfig.getSpec().getVersion() == null) {
            deployConfig.getSpec().setVersion(new SimpleDateFormat("ddHHmmss").format(new Date()));
        }
    }

    private void setDefaultConfigForNonServerless(
            ToolkitDeployConfig deployConfig,
            ToolkitPackageConfig packageConfig,
            MavenProject project) {
        //app
        //DefaultConfigBean.App app = config.getApp();
        //ToolkitDeployConfig.Spec spec = deployConfig.getSpec();
        if (packageConfig.getSpec().getPackageType() == null) {
            packageConfig.getSpec().setPackageType("url");
        }
        if (deployConfig.getSpec().getBatch() == null) {
            deployConfig.getSpec().setBatch(1);
        }
        if (deployConfig.getSpec().getGroupId() == null) {
            deployConfig.getSpec().setGroupId("all");
        }
        int batchWaitTime = (deployConfig.getSpec().getBatchWaitTime() != null) ? deployConfig.getSpec().getBatchWaitTime() : 0;
        if (deployConfig.getSpec().getStageTimeout() == null) {
            deployConfig.getSpec().setStageTimeout(batchWaitTime * 60 + 300);
        } else {
            int seconds = batchWaitTime * 60 + deployConfig.getSpec().getStageTimeout() * 60;
            deployConfig.getSpec().setStageTimeout(seconds);
        }
        if (deployConfig.getSpec().getVersion() == null) {
            String pv = String.format("%s (%s)", project.getVersion(), CommonUtils.getCurrentTime());
            deployConfig.getSpec().setVersion(pv);
        }
        deployConfig.getSpec().setServiceStageTimeout(deployConfig.getSpec().getStageTimeout());
        deployConfig.getSpec().setInstanceStageTimeout(deployConfig.getSpec().getStageTimeout());
    }

//    private void detectServerlessPackageType(
//            ToolkitDeployConfig deployConfig,
//            ToolkitPackageConfig packageConfig,
//            DefaultAcsClient acsClient) throws Exception {
//        GetServerlessAppConfigDetailRequest detailRequest = new GetServerlessAppConfigDetailRequest();
//        detailRequest.setAppId(deployConfig.getSpec().getTarget().getAppId());
//        GetServerlessAppConfigDetailResponse detailResponse = acsClient.getAcsResponse(detailRequest);
//        if (detailResponse.getCode() != 200) {
//            throw new Exception(String.format(
//                    "Unable to get serverless application detail by GetServerlessAppConfigDetailRequest, code: %d, msg: %s",
//                    detailResponse.getCode(), detailResponse.getMessage()));
//        }
//
//        String packageType = detailResponse.getData().getPackageType();
//        packageConfig.getSpec().setPackageType(packageType);
//    }

//    private void detectClusterType(DefaultConfigBean config, DefaultAcsClient acsClient) throws Exception {
//        if (!CommonUtils.isEmpty(config.getApp().getClusterType())) {
//            return;
//        }
//        logger.info("Detecting app cluster type...");
//
//        GetApplicationRequest request = new GetApplicationRequest();
//        request.setAppId(config.getApp().getAppId());
//        GetApplicationResponse response = acsClient.getAcsResponse(request);
//        if (response.getCode() == 200) {
//            Integer clusterType = response.getApplcation().getClusterType();
//            if (clusterType == null) {
//                clusterType = -1;
//            }
//            String typeStr;
//            switch (clusterType) {
//            case 0:
//                typeStr = DOCKER;
//                break;
//            case 1:
//                typeStr = SWAM;
//                break;
//            case 2:
//                typeStr = ECS;
//                break;
//            case 3:
//                typeStr = KUBERNETES;
//                break;
//            case 4:
//                typeStr = PANDORA;
//                break;
//            case 8:
//                typeStr = SERVERLESS;
//                break;
//            default:
//                typeStr = UNKNOWN;
//                break;
//            }
//            config.getApp().setClusterType(typeStr);
//            logger.info("App cluster type is: " + typeStr);
//        } else {
//            throw new Exception(String.format(
//                    "Unable to get application by GetApplicationRequest, code: %d, msg: %s",
//                    response.getCode(), response.getMessage()));
//        }
//    }
}
