package com.aliyun.manager;

import com.aliyun.Context;
import com.spotify.docker.client.DefaultDockerClient;
import com.spotify.docker.client.DockerClient;
import com.spotify.docker.client.ProgressHandler;
import com.spotify.docker.client.exceptions.DockerException;
import com.spotify.docker.client.messages.ProgressMessage;
import com.spotify.docker.client.messages.RegistryAuth;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;

import java.io.File;
import java.nio.file.Paths;
import java.util.concurrent.ArrayBlockingQueue;

public class ImageManager {

    private Log logger;

    public ImageManager() {
        this.logger = Context.getLogger();
    }

    public String buildAndUploadImage(
            String dockerfile,
            String repoAddress, String imageTag,
            String repoUser, String repoPassword,
            MavenProject project) throws Exception {
        DockerClient client = DefaultDockerClient.fromEnv().build();
        buildImage(client, dockerfile, repoAddress, imageTag, project);
        uploadImage(client, repoAddress, imageTag, repoUser, repoPassword, project);
        return getImageName(repoAddress, imageTag);
    }

    private void uploadImage(
            DockerClient client,
            String repoAddress, String imageTag,
            String repoUser, String repoPassword,
            MavenProject project) throws Exception {
        int index = repoAddress.indexOf("/");
        if (index < 0) {
            throw new Exception("Image repository is not correct: " + repoAddress);
        }
        String server = repoAddress.substring(0, index);

        RegistryAuth registryAuth = RegistryAuth.builder()
                .serverAddress(server)
                .username(repoUser)
                .password(repoPassword)
                .build();
        final ArrayBlockingQueue<String> imageDigestQueue = new ArrayBlockingQueue<>(1);
        String imageName = getImageName(repoAddress, imageTag);
        client.push(
                imageName,
                new ProgressHandler() {
                    @Override public void progress(ProgressMessage message) throws DockerException {
                        if (message.status() != null) {
                            logger.info(String.format("%s%s", (message.id() != null)?message.id()+": ":"", message.status()));
                        }
                        if (message.progress() != null) {
                            logger.info(message.progress());
                        }
                        if (message.digest() != null) {
                            imageDigestQueue.add(message.digest());
                        }
                    }
                }, registryAuth);
        String imageDigest = imageDigestQueue.poll();
        logger.info(String.format(
                "Upload image(%s) to repository successfully, digest is %s",
                imageName, imageDigest));
    }

    private String buildImage(
            DockerClient client,
            String dockerfile, String repoAddress, String imageTag,
            MavenProject project) throws Exception {

        logger.info("Building image...");

        File baseDir = project.getBasedir();
        logger.info("Use "+ dockerfile +" in " + baseDir);

        String imageId = client.build(
                Paths.get(baseDir.getPath()),
                getImageName(repoAddress, imageTag),
                dockerfile,
                new ProgressHandler() {
                    @Override
                    public void progress(ProgressMessage progressMessage) throws DockerException {
                        if (progressMessage.stream() != null) {
                            logger.info(progressMessage.stream().replaceAll("[\r\n]", ""));
                        }
                    }
                });
        logger.info("Build image successfully, image id: " + imageId);
        return imageId;
    }

    private String getImageName(String repoAddress, String imageTag) {
        return repoAddress + ":" + imageTag;
    }
}
