package com.aliyun.manager.deploy;

import com.aliyun.Context;
import com.aliyun.bean.common.OssProfile;
import com.aliyun.bean.common.ToolkitProfile;
import com.aliyun.bean.config.ToolkitDeployConfig;
import com.aliyun.bean.config.ToolkitPackageConfig;
import com.aliyun.enums.Constants;
import com.aliyun.manager.AppConfigManager;
import com.aliyun.manager.ChangeOrderManager;
import com.aliyun.manager.DeployPackageManager;
import com.aliyun.manager.ImageManager;
import com.aliyun.manager.ReportManager;
import com.aliyun.manager.SaeChangeOrderManager;
import com.aliyun.manager.UploadPackageManager;
import com.aliyun.utils.CommonUtils;
import com.aliyuncs.DefaultAcsClient;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;

public class DeployServerlessManager {
    private Log logger;
    private AppConfigManager appConfigManager;
    private UploadPackageManager uploadPackageManager;
    private ImageManager imageManager;
    private DeployPackageManager deployPackageManager;
    private SaeChangeOrderManager changeOrderManager;
    private ReportManager reportManager = new ReportManager();

    public DeployServerlessManager() {
        logger = Context.getLogger();
        appConfigManager = new AppConfigManager();
        uploadPackageManager = new UploadPackageManager(Context.getProject());
        imageManager = new ImageManager();
        deployPackageManager = new DeployPackageManager();
        changeOrderManager = new SaeChangeOrderManager();
        reportManager = new ReportManager();
    }

    public void deploy(ToolkitDeployConfig deployConfig, ToolkitPackageConfig packageConfig) throws Exception {
        MavenProject project = Context.getProject();
        DefaultAcsClient defaultAcsClient = Context.getAcsClient();

        checkParam(packageConfig);
        completeAppConfig(deployConfig, packageConfig, project, defaultAcsClient);
        uploadPackage(deployConfig, packageConfig);
        buildAndUploadImage(packageConfig, project);
        String changeOrderId = deployPackage(deployConfig, packageConfig, defaultAcsClient);
        traceChangeOrder(deployConfig, defaultAcsClient, changeOrderId);
        report(deployConfig, true);
    }

    private void checkParam(ToolkitPackageConfig packageConfig) throws Exception {
        if (packageConfig.getSpec().getPackageType() == null) {
            throw new Exception("Package type is missed in toolkit package config file.");
        }
        if (!Constants.IMAGE.equalsIgnoreCase(packageConfig.getSpec().getPackageType())
                && !Constants.WAR.equalsIgnoreCase(packageConfig.getSpec().getPackageType())
                && !Constants.FATJAR.equalsIgnoreCase(packageConfig.getSpec().getPackageType())) {
            throw new Exception("Package type should be Image/FatJar/War in toolkit package config file.");
        }
    }

    private void report(ToolkitDeployConfig deployConfig, boolean success) {
        reportManager.report(Context.getToolkitProfileConfig(), deployConfig, success);
    }

    private void traceChangeOrder(
            ToolkitDeployConfig deployConfig, DefaultAcsClient defaultAcsClient,
            String changeOrderId) throws Exception {
        changeOrderManager.trace(
                defaultAcsClient, changeOrderId,
                deployConfig.getSpec().getServiceStageTimeout(),
                deployConfig.getSpec().getInstanceStageTimeout());
    }

    private String deployPackage(ToolkitDeployConfig deployConfig, ToolkitPackageConfig packageConfig, DefaultAcsClient defaultAcsClient) throws Exception {
        return deployPackageManager.deployPackage(defaultAcsClient, deployConfig, packageConfig);
    }

    private void buildAndUploadImage(ToolkitPackageConfig packageConfig,
            MavenProject project) throws Exception {
        if (!Constants.IMAGE.equalsIgnoreCase(packageConfig.getSpec().getPackageType())) {
            return;
        }

        if (packageConfig.getSpec().getImageUrl() != null) {
            logger.info("Image url is set, skip building image.");
            return;
        }

        if (packageConfig.getSpec().getBuild() == null) {
            throw new Exception("Failed to build image: imageUrl or Build is not set in toolkit package config file.");
        }

        if (packageConfig.getSpec().getBuild().getDocker() == null) {
            throw new Exception("Failed to build image: imageUrl or Docker is not set in toolkit package config file.");
        }

        if (packageConfig.getSpec().getBuild().getDocker().getImageRepoAddress() == null) {
            throw new Exception("Failed to build image: imageUrl or imageRepoAddress is not set in toolkit package config file.");
        }

        if (packageConfig.getSpec().getBuild().getDocker().getImageTag() == null) {
            throw new Exception("Failed to build image: imageUrl or imageTag is not set in toolkit package config file.");
        }


        String dockerfile = packageConfig.getSpec().getBuild().getDocker().getDockerfile();
        if (dockerfile == null) {
            dockerfile = Constants.DOCKERFILE;
        }
        logger.info(String.format(
                "Use %s to build image in current build repository(%s)",
                dockerfile, project.getBasedir().getPath()));
        packageConfig.getSpec().setImageUrl(imageManager.buildAndUploadImage(
                dockerfile,
                packageConfig.getSpec().getBuild().getDocker().getImageRepoAddress(),
                packageConfig.getSpec().getBuild().getDocker().getImageTag(),
                packageConfig.getSpec().getBuild().getDocker().getImageRepoUser(),
                packageConfig.getSpec().getBuild().getDocker().getImageRepoPassword(),
                project));
    }

    private void completeAppConfig(
            ToolkitDeployConfig deployConfig,
            ToolkitPackageConfig packageConfig,
            MavenProject project, DefaultAcsClient defaultAcsClient) throws Exception {
        appConfigManager.complete(deployConfig, packageConfig, project, defaultAcsClient);
    }

    private void uploadPackage(ToolkitDeployConfig deployConfig, ToolkitPackageConfig packageConfig) throws Exception {
        if (Constants.IMAGE.equalsIgnoreCase(packageConfig.getSpec().getPackageType())) {
            return;
        }
        if (packageConfig.getSpec().getPackageUrl() != null) {
            return;
        }

        //ToolkitDeployConfig.Spec spec = deployConfig.getSpec();
        ToolkitProfile toolkitProfile = Context.getToolkitProfileConfig();
        String packageVersion = deployConfig.getSpec().getVersion();
        if (packageVersion == null) {
            //TODO 在更早前设置此值
            packageVersion = CommonUtils.getCurrentTime();
        }

        //直接看到serverless就替换product
        toolkitProfile.setProduct(CommonUtils.Product.Sae.name());
        String appId = null;
        if (CommonUtils.isNotEmpty(deployConfig) && CommonUtils.isNotEmpty(deployConfig.getSpec()) &&
            CommonUtils.isNotEmpty(deployConfig.getSpec().getTarget())) {
            appId = deployConfig.getSpec().getTarget().getAppId();
        }
        String downloadUrl = uploadPackageManager.uploadPackage(toolkitProfile, null,
            appId, packageVersion);
        packageConfig.getSpec().setPackageUrl(downloadUrl);
    }
}
