package com.aliyun.manager.deploy;

import com.aliyun.Context;
import com.aliyun.bean.common.OssProfile;
import com.aliyun.bean.common.ToolkitProfile;
import com.aliyun.bean.config.ToolkitPackageConfig;
import com.aliyun.bean.config.ToolkitDeployConfig;
import com.aliyun.enums.Constants;
import com.aliyun.manager.*;
import com.aliyun.utils.CommonUtils;
import com.aliyuncs.DefaultAcsClient;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;

public class ToolkitDeployManager {
    private Log logger;
    private AppConfigManager appConfigManager;
    private UploadPackageManager uploadPackageManager;
    private ImageManager imageManager;
    private DeployPackageManager deployPackageManager;
    private ChangeOrderManager changeOrderManager;
    private ReportManager reportManager;
    private DeployServerlessManager deployServerlessManager;

    public ToolkitDeployManager() {
        logger = Context.getLogger();
        appConfigManager = new AppConfigManager();
        uploadPackageManager = new UploadPackageManager(Context.getProject());
        imageManager = new ImageManager();
        deployPackageManager = new DeployPackageManager();
        changeOrderManager = new ChangeOrderManager();
        reportManager = new ReportManager();
        deployServerlessManager = new DeployServerlessManager();
    }

    public void deploy(ToolkitDeployConfig deployConfig, ToolkitPackageConfig packageConfig) throws Exception {
        ObjectMapper jsonMapper = new ObjectMapper();
        jsonMapper.configure(SerializationFeature.INDENT_OUTPUT,true);
        logger.info("deployConfig = \n" + jsonMapper.writeValueAsString(deployConfig)+"\n");
        logger.info("packageConfig = \n" + jsonMapper.writeValueAsString(packageConfig)+"\n");

        // 从此走sae分支
        if (Constants.SERVERLESS.equalsIgnoreCase(deployConfig.getSpec().getType())) {
            deployServerlessManager.deploy(deployConfig, packageConfig);
            return;
        }

        MavenProject project = Context.getProject();
        DefaultAcsClient defaultAcsClient = Context.getAcsClient();

        checkParam(deployConfig, packageConfig);
        completeAppConfig(deployConfig, packageConfig, project, defaultAcsClient);
        uploadPackage(deployConfig, packageConfig);
        buildAndUploadImage(deployConfig, packageConfig, project);
        String changeOrderId = deployPackage(deployConfig, packageConfig, defaultAcsClient);
        traceChangeOrder(deployConfig, defaultAcsClient, changeOrderId);
        report(deployConfig, true);
    }

    private void checkParam(ToolkitDeployConfig deployConfig, ToolkitPackageConfig packageConfig) throws Exception {
        if (packageConfig.getSpec().getPackageType() == null) {
            throw new Exception("Package type is missed in toolkit package config file.");
        }
        if (Constants.KUBERNETES.equalsIgnoreCase(deployConfig.getSpec().getType())
                && !Constants.IMAGE.equalsIgnoreCase(packageConfig.getSpec().getPackageType())
                && !Constants.WAR.equalsIgnoreCase(packageConfig.getSpec().getPackageType())
                && !Constants.FATJAR.equalsIgnoreCase(packageConfig.getSpec().getPackageType())) {
            throw new Exception("Package type should be Image/FatJar/War in toolkit package config file.");
        }
    }

    private void report(ToolkitDeployConfig deployConfig, boolean success) {
        reportManager.report(Context.getToolkitProfileConfig(), deployConfig, success);
    }

    private void traceChangeOrder(
            ToolkitDeployConfig deployConfig, DefaultAcsClient defaultAcsClient,
            String changeOrderId) throws Exception {
        changeOrderManager.trace(
                defaultAcsClient, changeOrderId,
                deployConfig.getSpec().getServiceStageTimeout(),
                deployConfig.getSpec().getInstanceStageTimeout());
    }

    private String deployPackage(ToolkitDeployConfig deployConfig, ToolkitPackageConfig packageConfig, DefaultAcsClient defaultAcsClient) throws Exception {
        return deployPackageManager.deployPackage(defaultAcsClient, deployConfig, packageConfig);
    }

    private void buildAndUploadImage(
            ToolkitDeployConfig deployConfig,
            ToolkitPackageConfig packageConfig,
            MavenProject project) throws Exception {
        // 对k8s镜像方式的支持
        if (!Constants.KUBERNETES.equals(deployConfig.getSpec().getType())) {
            return;
        }

        if (!Constants.IMAGE.equalsIgnoreCase(packageConfig.getSpec().getPackageType())) {
            return;
        }

        if (packageConfig.getSpec().getImageUrl() != null) {
            logger.info("Image url is set, skip building image.");
            return;
        }

        if (packageConfig.getSpec().getBuild() == null) {
            throw new Exception("Failed to build image: imageUrl or Build is not set in toolkit package config file.");
        }

        if (packageConfig.getSpec().getBuild().getDocker() == null) {
            throw new Exception("Failed to build image: imageUrl or Docker is not set in toolkit package config file.");
        }

        if (packageConfig.getSpec().getBuild().getDocker().getImageRepoAddress() == null) {
            throw new Exception("Failed to build image: imageUrl or imageRepoAddress is not set in toolkit package config file.");
        }

        if (packageConfig.getSpec().getBuild().getDocker().getImageTag() == null) {
            throw new Exception("Failed to build image: imageUrl or imageTag is not set in toolkit package config file.");
        }


        String dockerfile = packageConfig.getSpec().getBuild().getDocker().getDockerfile();
        if (dockerfile == null) {
            dockerfile = Constants.DOCKERFILE;
        }
        logger.info(String.format(
                "Use %s to build image in current build repository(%s)",
                dockerfile, project.getBasedir().getPath()));
        String imageUrl = imageManager.buildAndUploadImage(
                dockerfile,
                packageConfig.getSpec().getBuild().getDocker().getImageRepoAddress(),
                packageConfig.getSpec().getBuild().getDocker().getImageTag(),
                packageConfig.getSpec().getBuild().getDocker().getImageRepoUser(),
                packageConfig.getSpec().getBuild().getDocker().getImageRepoPassword(),
                project);
        // 回写到packageConfig中
        packageConfig.getSpec().setImageUrl(imageUrl);
    }

    private void completeAppConfig(
            ToolkitDeployConfig deployConfig,
            ToolkitPackageConfig packageConfig,
            MavenProject project, DefaultAcsClient defaultAcsClient) throws Exception {
        appConfigManager.complete(deployConfig, packageConfig, project, defaultAcsClient);
    }

    private void uploadPackage(ToolkitDeployConfig deployConfig, ToolkitPackageConfig packageConfig) throws Exception {
        if (Constants.URL.equals(packageConfig.getSpec().getPackageType())
                && packageConfig.getSpec().getPackageUrl() != null) {
            return;
        }

        // 过滤掉k8s镜像的
        if (Constants.IMAGE.equals(packageConfig.getSpec().getPackageType())) {
            return;
        }

        //ToolkitDeployConfig.Spec spec = deployConfig.getSpec();
        ToolkitProfile toolkitProfile = Context.getToolkitProfileConfig();
        OssProfile customOss = packageConfig.getSpec().getBuild().getOss();
        String appId = deployConfig.getSpec().getTarget().getAppId();
        String packageVersion = deployConfig.getSpec().getVersion();
        if (packageVersion == null) {
            //TODO 在更早前设置此值
            packageVersion = CommonUtils.getCurrentTime();
        }
        //System.out.println(customOss);
        String downloadUrl = uploadPackageManager.uploadPackage(toolkitProfile, customOss, appId, packageVersion);
        packageConfig.getSpec().setPackageUrl(downloadUrl);
    }
}
