package com.aliyun.utils;

import com.aliyun.Context;
import com.aliyuncs.AcsRequest;
import com.aliyuncs.AcsResponse;
import com.aliyuncs.IAcsClient;
import com.aliyuncs.edas.model.v20170801.GetChangeOrderInfoRequest;
import com.aliyuncs.exceptions.ClientException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import lombok.SneakyThrows;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;

import java.io.Closeable;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

public class HttpUtils {
    private static final List<String> RETRY_ERROR_CODE_LIST = Arrays.asList("500","503");
    private static final AtomicInteger INCR = new AtomicInteger();

    @SneakyThrows
    public static <RESPONSE extends AcsResponse>RESPONSE getAcsResponseIfRetry(IAcsClient client, AcsRequest<RESPONSE> request) throws ClientException {
        RESPONSE response = null;
        int retryCount = 0;
        if(!(request instanceof GetChangeOrderInfoRequest)) {
            ObjectMapper jsonMapper = new ObjectMapper();
            jsonMapper.configure(SerializationFeature.INDENT_OUTPUT, true);
            Context.getLogger().info("request["+INCR.incrementAndGet()+"] = " + request.getClass().getSimpleName() + "\n" + jsonMapper.writeValueAsString(request) + "\n");
        }
        do {
            try {
                request.setSysConnectTimeout(5000);
                request.setSysReadTimeout(5000);
                response = client.getAcsResponse(request);
            }catch (com.aliyuncs.exceptions.ServerException e){
                //阿里云服务熔断, 1秒后重试. https://www.alibabacloud.com/help/zh/doc-detail/43906.htm
                if(RETRY_ERROR_CODE_LIST.contains(e.getErrCode())){
                    Thread.sleep(3000);
                }
                Context.getLogger().warn(String.format("%s server error retryCount(%d). msg=(%s)",request.getClass().getSimpleName(),retryCount,e.toString()),e);
            }catch (com.aliyuncs.exceptions.ClientException e){
                if(e.getCause()!= null && e.getCause() instanceof IOException){
                    retryCount++;
                    Context.getLogger().warn(String.format("%s error retryCount(%d). msg=(%s)",request.getClass().getSimpleName(),retryCount,e.toString()));
                }else {
                    throw e;
                }
            }
        }while (response == null);
        return response;
    }

    /**
     * POST文本数据
     * @param url：目的地址
     * @param body：body数据
     * @param defaultTimeout: 默认超时，单位毫秒
     */
    public static void postJson(String url, String body, int defaultTimeout) {
        CloseableHttpClient client = null;
        CloseableHttpResponse response = null;
        try {
            //client
            RequestConfig config = RequestConfig.custom()
                    .setConnectTimeout(defaultTimeout)
                    .setConnectionRequestTimeout(defaultTimeout)
                    .setSocketTimeout(defaultTimeout).build();
            client = HttpClientBuilder.create().setDefaultRequestConfig(config).build();
            //http post
            HttpPost httpPost = new HttpPost(url);
            StringEntity entity = new StringEntity(body);
            httpPost.setHeader("Content-type", "application/json");
            httpPost.setEntity(entity);
            //execute
            response = client.execute(httpPost);
        } catch (Throwable t) {
            //do nothing
        } finally {
            close(response);
            close(client);
        }
    }

    /**
     * POST文本数据
     * @param url：目的地址
     * @param body：body数据
     * @param defaultTimeout: 默认超时，单位毫秒
     */
    public static void postPlainText(String url, String body, int defaultTimeout) {
        CloseableHttpClient client = null;
        CloseableHttpResponse response = null;
        try {
            //client
            RequestConfig config = RequestConfig.custom()
                    .setConnectTimeout(defaultTimeout)
                    .setConnectionRequestTimeout(defaultTimeout)
                    .setSocketTimeout(defaultTimeout).build();
            client = HttpClientBuilder.create().setDefaultRequestConfig(config).build();
            //http post
            HttpPost httpPost = new HttpPost(url);
            StringEntity entity = new StringEntity(body);
            httpPost.setHeader("Content-type", "text/plain");
            httpPost.setEntity(entity);
            //execute
            response = client.execute(httpPost);
        } catch (Throwable t) {
            //do nothing
        } finally {
            close(response);
            close(client);
        }
    }

    private static void close(Closeable closeable) {
        try {
            if (closeable != null) {
                closeable.close();
            }
        } catch (Throwable t) {
            //do nothing
        }
    }
}
